// AR-B6002.cpp : Defines the entry point for the DLL application.
//

#include "stdafx.h"
#include "AR-B6002.h"

/* For Power Subsystem */
#include "pwr_subsystem.h"
#include "SerialPort.h"

/* For CAN bus */
#include "CAN_LIB.h"
#include "errcode.h"
#include "Ioctls.h"


static DEVINFO DevInfo = {
	INVALID_HANDLE_VALUE,
	false,
};


/* For GPIO and Watchdog */
#include <stdio.h>
#include "sio_access.h"
#include "errno.h"
#include "winio.h"


#ifdef _MANAGED
#pragma managed(push, off)
#endif



#ifdef _MANAGED
#pragma managed(pop)
#endif

/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* ------------------------------------------ Power Subsystem APIs ------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
CSerialPort comport;  //com port object


char doHandshake(void)
{
  int hshakeOk = 0;
  int test;
  BYTE bRcv;
  for(test=0;test<50;test++) {
    Sleep(2);
    comport.ReadByte(bRcv);
    if(bRcv == 0xAA)
      break;
  }
  if (0xAA==bRcv)
    hshakeOk = 1;

  if (hshakeOk)
    comport.WriteByte(0x55);				/* send ACK */

  return hshakeOk;
}

BOOL initComPort(CSerialPort &comport)
{
    if(comport.OpenPort(6))	{
		if(comport.ConfigurePort(CBR_19200,8,false,NOPARITY,ONESTOPBIT)) {
			comport.SetCommunicationTimeouts(1000 ,1000 ,1000,1000,1000);
			return(TRUE);
		} else {
			return FALSE;
		}
	} else {
			return FALSE;
	}
}


ARB6002_API i32 getIgnStatus(u8 *ignStatus)
{
	BOOL result = FALSE;
	BYTE hshake,errCode;

	if( comport.hComm == 0x0 ) {
		result = initComPort(comport);
		if( result == FALSE )
			return(-1);
	}

    
    /* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_GET_IGN_STS);

    /* Step 2, do handshake */
    hshake = doHandshake();

    /* 1st checkpoint */ 
    if (hshake != 1)
      return(-1);

    comport.ReadByte(*ignStatus);
    comport.ReadByte(errCode); //read errCode
   
	return(0);
}

/* Set the delay time in unit 'seconds'. */
ARB6002_API i32 setSoftOffDelayS(u32 val)
{
	BOOL result = FALSE;
	BYTE hshake,errCode;
	    
    unsigned char data[3];
    int i;

	/*  0 <= val <= 255 */
	if( val > 255 )
		return(-1);

	if( comport.hComm == 0x0 ) {
		result = initComPort(comport);
		if( result == FALSE )
			return(-1);
	}


    /* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_SET_SOFTOFF);
    /* Step 2, do handshake */
    hshake = doHandshake();		
    /* 1st checkpoint */ 
    if (hshake != 1)
      return(-1);

    val*=10;  //change 1s unit into 100ms unit
    data[0] = (unsigned char)(val&0xFF);
    data[1] = (unsigned char)((val>>8)&0xFF);
    data[2] = (unsigned char)((val>>16)&0xFF);

	for(i = 0 ; i < 3 ; i++) { //send 3 byte softoff data
      comport.WriteByte(data[i]);
      Sleep(1);
    }
    comport.ReadByte(errCode); //read errCode
    
	return(0);
}

/* Set the delay time in unit 'minutes'. */
ARB6002_API i32 setSoftOffDelayM(u32 val)
{
	BOOL result = FALSE;
	BYTE hshake,errCode;
	    
    unsigned char data[3];
    int i;

	/*  0 <= val <= 255 */
	if( val > 255 )
		return(-1);

	if( comport.hComm == 0x0 ) {
		result = initComPort(comport);
		if( result == FALSE )
			return(-1);
	}


    /* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_SET_SOFTOFF);
    /* Step 2, do handshake */
    hshake = doHandshake();		
    /* 1st checkpoint */ 
    if (hshake != 1)
      return(-1);

    val=val*60*10;  //change 1s unit into 100ms unit
    data[0] = (unsigned char)(val&0xFF);
    data[1] = (unsigned char)((val>>8)&0xFF);
    data[2] = (unsigned char)((val>>16)&0xFF);

	for(i = 0 ; i < 3 ; i++) { //send 3 byte softoff data
      comport.WriteByte(data[i]);
      Sleep(1);
    }
    comport.ReadByte(errCode); //read errCode
    
	return(0);
}

/* Set the delay time in unit 'seconds'. */
ARB6002_API i32 setHardOffDelayS(u32 val)
{
	BOOL result = FALSE;
	BYTE hshake, errCode;

	unsigned char data[3];
    int i;

	/*  0 <= val <= 255 */
	if( val > 255 )
		return(-1);

	if( comport.hComm == 0x0 ) {
		result = initComPort(comport);
		if( result == FALSE )
			return(-1);
	}


    /* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_SET_HARDOFF);
    /* Step 2, do handshake */
    hshake = doHandshake();		
    /* 1st checkpoint */ 
    if (hshake != 1)
      return FALSE;
    val*=10; //change 1s unit into 100ms unit
    data[0] = (unsigned char)(val & 0xFF);
    data[1] = (unsigned char)((val>>8) & 0xFF);
    data[2] = (unsigned char)((val>>16) & 0xFF);

	for(i = 0; i < 3; i++) { //send 3 byte hardoff data
      comport.WriteByte(data[i]);
      Sleep(1);
    }
    comport.ReadByte(errCode); //read errCode

	return(0);
}

/* Set the delay time in unit 'minutes'. */
ARB6002_API i32 setHardOffDelayM(u32 val)
{
	BOOL result = FALSE;
	BYTE hshake, errCode;

	unsigned char data[3];
    int i;

	/*  0 <= val <= 255 */
	if( val > 255 )
		return(-1);

	if( comport.hComm == 0x0 ) {
		result = initComPort(comport);
		if( result == FALSE )
			return(-1);
	}


    /* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_SET_HARDOFF);
    /* Step 2, do handshake */
    hshake = doHandshake();		
    /* 1st checkpoint */ 
    if (hshake != 1)
      return FALSE;
    val=val*60*10; //change 1s unit into 100ms unit
    data[0]=(unsigned char)(val&0xFF);
    data[1]=(unsigned char)((val>>8)&0xFF);
    data[2]=(unsigned char)((val>>16)&0xFF);

	for(i = 0 ; i < 3 ; i++) { //send 3 byte hardoff data
      comport.WriteByte(data[i]);
      Sleep(1);
    }
    comport.ReadByte(errCode); //read errCode

	return(0);
}

ARB6002_API i32 setPowerOnMode(u8 powerOnMode)
{
	BOOL result = FALSE;
	BYTE hshake, errCode;

	if( comport.hComm == 0x0 ) {
	  result = initComPort(comport);
	  if( result == FALSE )
		return(-1);
	}

	/* Step 1, Seed Command*/ 
    comport.WriteByte(CMD_SET_PWR_MOD);
    /* Step 2, do handshake */
    hshake = doHandshake();		
    /* 1st checkpoint */ 
    if (hshake != 1)
      return(-1);

    comport.WriteByte(powerOnMode);
    comport.ReadByte(errCode); //read errCode
    
	return(0);
}

ARB6002_API i32 getSoftOffDelay(u32* timeDelay)
{
	BOOL result = FALSE;
	BYTE hshake, errCode, value;
	int i;
	unsigned char val[3];

	if( comport.hComm == 0x0 ) {
	  result = initComPort(comport);
	  if( result == FALSE )
		return(-1);
	}

   /* Step 1, Seed Command*/ 
   comport.WriteByte(CMD_GET_SOFTOFF);
   /* Step 2, do handshake */
   hshake = doHandshake();		
   /* 1st checkpoint */ 
   if(hshake != 1)
     return(-1);

   //GET VER FOR VALUE 2BYTE DATA	
   for(i = 0 ; i < 3 ; i++) {
     comport.ReadByte(value);
     val[i] = value;
   }

   //GET END
   comport.ReadByte(errCode); //read errCode
   *timeDelay = (val[2]<<16)+(val[1]<<8)+val[0];
   *timeDelay = (*timeDelay)/10; //change 100ms unit to 1s unit
   return(0);
}

ARB6002_API i32 getHardOffDelay(u32* timeDelay)
{
  BOOL result = FALSE;
  BYTE hshake, errCode, value;
  unsigned char val[3];
  int i;
	
  if( comport.hComm == 0x0 ) {
    result = initComPort(comport);
    if( result == FALSE )
  	return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_GET_HARDOFF);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);

  for(i = 0 ; i < 3 ; i++) {
    comport.ReadByte(value);
    val[i] = value;
  }
  comport.ReadByte(errCode); //read errCode
  *timeDelay = (val[2]<<16)+(val[1]<<8)+val[0];
  *timeDelay = *timeDelay/10; //change 100ms unit to 1s unit

  return(0);
}

ARB6002_API i32 getPowerOnMode(u8 *powerOnMode)
{
  BOOL result = FALSE;
  BYTE hshake, errCode;

  if( comport.hComm == 0x0 ) {
  result = initComPort(comport);
  if( result == FALSE )
	return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_GET_PWR_MOD);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);

  comport.ReadByte(*powerOnMode);
  comport.ReadByte(errCode); //read errCode
  return(0);

}

ARB6002_API i32 getBattVolt(float *volt)
{
  BOOL result = FALSE;
  BYTE hshake, errCode, value;
  int i;
  unsigned char bat[2];
  
  if( comport.hComm == 0x0 ) {
    result = initComPort(comport);

    if( result == FALSE )
	  return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_GET_BAT_VOL);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);

  for(i = 0 ; i < 2 ; i++) {
    comport.ReadByte(value);
    bat[i] = value;
  }
  comport.ReadByte(errCode); //read errCode
  *volt = bat[1]*256;
  *volt += bat[0];
  *volt = (*volt)*6*5/1024;

  return(0);
}

ARB6002_API i32 getPicFwVer(PicInfo *ver)
{
  BOOL result = FALSE;
  BYTE *temp = (unsigned char *)ver;
  BYTE hshake, errCode, value;
  int i;

  if( comport.hComm == 0x0 ) {
    result = initComPort(comport);
    
	if( result == FALSE )
	  return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_GET_PIC_VER);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);

  for(i = 0 ; i < 9 ; i++) {
    comport.ReadByte(value);
    temp[i] = value;
  }
  comport.ReadByte(errCode); //read errCode
  return(0);
}

ARB6002_API i32 getPicMode(u8 *mode)
{
  BOOL result = FALSE;
  BYTE hshake, errCode;
	  
  if( comport.hComm == 0x0 ) {
  result = initComPort(comport);
  if( result == FALSE )
	return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_GET_PIC_MOD);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);
  comport.ReadByte(*mode);
  comport.ReadByte(errCode); //read errCode
  return(0);
}

ARB6002_API i32 setPicDefault(void)
{
  BOOL result = FALSE;
  BYTE hshake, errCode;
		  
  if( comport.hComm == 0x0 ) {
  result = initComPort(comport);
  if( result == FALSE )
	return(-1);
  }

  /* Step 1, Seed Command*/ 
  comport.WriteByte(CMD_SET_PIC_DEFAULT);
  /* Step 2, do handshake */
  hshake = doHandshake();		
  /* 1st checkpoint */ 
  if (hshake != 1)
    return(-1);

  comport.ReadByte(errCode); //read errCode

  return(0);
}


/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* ------------------------------------------- CAN Bus APIs -------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */


/*
*  REGISTERCARD FUN FOR switch cardnumber 
*  INPUT  : CARDNUM
*  Install Define Setup
*/

unsigned int registerCard(void)
{
  HANDLE hARBPort;
  hARBPort = INVALID_HANDLE_VALUE;

  hARBPort=CreateFileA("\\\\.\\can", GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  DevInfo.drv_handle = hARBPort;
  if (hARBPort == INVALID_HANDLE_VALUE) {
    DevInfo.is_device_open = FALSE;
    return ERRMSG(0, ERROR_GEN_DRIVER_OPEN); // This function is used internally.
	                                         // There is no API ERROR code defined.
  }
  DevInfo.is_device_open = TRUE;

  return ERROR_API_SUCC;
}

/*
* FOR Release Card 
* INPUT  : IO ADDRESS 
* OUTPTU : ERROR CODE
*/
unsigned int releaseCard(void)
{
  if( DevInfo.is_device_open == FALSE )
    return ERRMSG(0,ERROR_GEN_DEVICE_OPEN);// Device not open
  
  //CloseHandle(DevInfo.drv_handle);

  DevInfo.is_device_open = FALSE;

  return ERROR_API_SUCC;
}

ARB6002_API unsigned int sendCanMessages(canmsg_t* buf, u8 count)
{
	DWORD written;
	int ret;
	unsigned int result = ERROR_API_SUCC;
	canmsg_t* pkgBeingSent = buf;
	int i;

	/* Make sure the CAN device has been opened. */
	if(DevInfo.drv_handle == INVALID_HANDLE_VALUE)
		result=registerCard();

	if(result != ERROR_API_SUCC )
		return ERRMSG(ERROR_API_CANSENDMESSAGES, result);

	for(i=0; i<count; i++) {
		ret = WriteFile(DevInfo.drv_handle, pkgBeingSent, sizeof(canmsg_t), &written, NULL);
		pkgBeingSent++;
	}

	return ERROR_API_SUCC;
}

ARB6002_API unsigned int getCanMessages(canmsg_t* buf, u8 count)
{
	DWORD read;
	int ret;
	unsigned int result=ERROR_API_SUCC;

	/* Make sure the CAN device has been opened. */
	if(DevInfo.drv_handle == INVALID_HANDLE_VALUE)
		result=registerCard();

	if(result != ERROR_API_SUCC )
		return ERRMSG(ERROR_API_CANGETMESSAGES, ERROR_GEN_DEVICE_FAIL);
	
	ret = ReadFile(DevInfo.drv_handle,buf,count*sizeof(canmsg_t),&read,NULL);
	
	return ERROR_API_SUCC;
}

ARB6002_API unsigned int configCan(i32 baud)
{
	Config_par_t canConfig;
	Command_par_t canCmd;
	DWORD BytesReturned;
	unsigned int result=ERROR_API_SUCC;

	/* Make sure the CAN device has been opened. */
	if(DevInfo.drv_handle == INVALID_HANDLE_VALUE)
		result=registerCard();

	if(result != ERROR_API_SUCC )
		return ERRMSG(ERROR_API_CANCONFIG, ERROR_GEN_DEVICE_FAIL);
	
	if( (baud != 10) && (baud != 20) && (baud != 50) && (baud != 100) && (baud != 125) && (baud != 250) && (baud != 500) && 
		(baud != 800) && (baud != 1000) )
		return ERRMSG(ERROR_API_CANCONFIG, ERROR_GEN_INPUT_DATA);
	
	canCmd.cmd = CMD_STOP;
    DeviceIoControl(DevInfo.drv_handle, (DWORD)ARBIOC_CAN_COMMAND, &canCmd, sizeof(Command_par_t),
		&canCmd, sizeof(Command_par_t), &BytesReturned, NULL);
	
    canConfig.target = CONF_TIMING; 
    canConfig.val1   = baud;
    DeviceIoControl(DevInfo.drv_handle, (DWORD)ARBIOC_CAN_CONFIG, &canConfig, sizeof(Config_par_t),
		&canConfig, sizeof(Config_par_t), &BytesReturned, NULL);
	
	canCmd.cmd = CMD_START;
	DeviceIoControl(DevInfo.drv_handle, (DWORD)ARBIOC_CAN_COMMAND, &canCmd, sizeof(Command_par_t),
		&canCmd, sizeof(Command_par_t), &BytesReturned, NULL);
	
	return ERROR_API_SUCC;
}





/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* -------------------------------------- GPIO and Watchdog APIs --------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------------------- */

BOOL winIoReady = FALSE;

/* Read one byte from SIO register */
static unsigned char readSioByte(unsigned short reg);

/* Write one byte to SIO register */
static void writeSioByte(unsigned short reg, unsigned char data);

/* Read one byte frome a register in SIO Logic Device */
static unsigned char readLogDevByte(unsigned char devNum, unsigned short reg);

/* Write one byte to a register in SIO Logic Device */
static void writeLogDevByte(unsigned char devNum, unsigned short reg, unsigned char data);

/* Configure GPIO30 ~ GPIO37 to the expected operating status */
static void setupGpio(void);

/* Reset Watchdog timer */
static void resetWtd(void);

/* Configure Watchdog to the expected operating status */
static void setupWtd(void);

static unsigned char regAry[4];

static void clearRegAry(void);

void clearRegAry(void)
{
	int i;
	for(i=0; i<4; i++)
		regAry[i] = 0x0;
	return;
}

unsigned char readSioByte(unsigned short reg)
{
  
  /* Enter the Extneded Function Mode */
  //outb( 0x87, SIO_EFER );
  //outb( 0x87, SIO_EFER );

  bool result = false;

  result = SetPortVal(SIO_EFER, 0x87, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFER, 0x87, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFIR, reg, 1);
  if( result != true )
	  return(ERR_SETPORT);

  clearRegAry();
  result = GetPortVal(SIO_EFDR, (PDWORD) regAry, 1);
  if( result != true )
	  return(ERR_GETPORT); 

  /* Exit the Extneded Function Mode */
  //outb( 0xAA, SIO_EFER);
  result = SetPortVal(SIO_EFER, 0xAA, 1);

  return(regAry[0]);
}

void writeSioByte(unsigned short reg, unsigned char data)
{
  bool result = false;


  result = SetPortVal(SIO_EFER, 0x87, 1);

  result = SetPortVal(SIO_EFER, 0x87, 1);

  result = SetPortVal(SIO_EFIR, reg, 1);

  result = SetPortVal(SIO_EFDR, data, 1);

  result = SetPortVal(SIO_EFER, 0xaa, 1);

  return;
}


unsigned char readLogDevByte(unsigned char devNum, unsigned short reg)
{
  bool result = false;

  result = SetPortVal(SIO_EFER, 0x87, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFER, 0x87, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFIR, SIO_SEL_DEV, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFDR, devNum, 1);
  if( result != true )
	  return(ERR_SETPORT);

  result = SetPortVal(SIO_EFIR, reg, 1);
  if( result != true )
	  return(ERR_SETPORT);

  clearRegAry();
  result = GetPortVal(SIO_EFDR, (PDWORD) regAry, 1);
  if( result != true )
	  return(ERR_GETPORT);

  result = SetPortVal(SIO_EFER, 0xaa, 1);
  if( result != true )
	  return(ERR_SETPORT);

  return((unsigned char) regAry[0]);
}


void writeLogDevByte(unsigned char devNum, unsigned short reg, unsigned char data)
{
  bool result = false;

  result = SetPortVal(SIO_EFER, 0x87, 1);

  result = SetPortVal(SIO_EFER, 0x87, 1);

  result = SetPortVal(SIO_EFIR, SIO_SEL_DEV, 1);

  result = SetPortVal(SIO_EFDR, devNum, 1);

  result = SetPortVal(SIO_EFIR, reg, 1);

  result = SetPortVal(SIO_EFDR, data, 1);

  result = SetPortVal(SIO_EFER, 0xaa, 1);

  return;
}

/* Configure pin GP32, GP33, GP34 as GPIO */
void setupGpio(void)
{
  unsigned char retval = 0x0;

  /* Configure GP32, GP33, GP34 as GPIO */
  retval = readSioByte(SIO_MFUNC_PIN_REG1);
  retval = retval & ~0xe0; // mask out bit 5~7
  writeSioByte(SIO_MFUNC_PIN_REG1, retval);


  /* Disable eventroute of GP30, GP31, GP35 to PSOUT# and PME# */
  retval = readLogDevByte(ACPI_DEV, 0xfe);  
  retval = retval & SIO_GPIO_EVTROUT;
  writeLogDevByte(ACPI_DEV, 0xfe, retval);

  /* Configure pin GP30 ~ GP37 as GPIO, not SUSLED */
  writeLogDevByte(GPIO_DEV, SIO_GPIO_MFUNC, 0x0);

  /* Make sure the signals at GP30 ~ GP37 are not inverted. */
  writeLogDevByte(GPIO_DEV, SIO_GPIO_IVRSN, 0x0);

  /* Activate GPIO3 */
  retval = readLogDevByte(GPIO_DEV, SIO_GPIO_ACT_REG); 
  retval = retval | SIO_GPIO_ACT;
  writeLogDevByte(GPIO_DEV, SIO_GPIO_ACT_REG, retval);

  /* Configure GP30 ~ GP33 as Output, GP34 ~ GP37 as Input */
  writeLogDevByte(GPIO_DEV, SIO_GPIO_CONF, SIO_GPIO_SET_IO);

  return;
}

void setupWtd(void)
{
  unsigned char retval = 0x0;

  /* Configure pin 77 as WDTO# */
  retval = readSioByte(SIO_MFUNC_PIN_REG2);
  retval = retval & SIO_WTD_PINEAB;
  writeSioByte(SIO_MFUNC_PIN_REG2, retval);

  /* Set WDTO# active */
  retval = readLogDevByte(WTD_DEV, SIO_WTD_ACT_REG);
  retval = retval | SIO_WTD_ACTIVE;
  writeLogDevByte(WTD_DEV, SIO_WTD_ACT_REG, retval);

  /* Set up timer unit to second */
  retval = readLogDevByte(WTD_DEV, SIO_WTD_CONTROL);
  retval = retval & SIO_WTD_SECOND;
  writeLogDevByte(WTD_DEV, SIO_WTD_CONTROL, retval);

  return;
}

/* Restore WDTO# if it has ever been triggered */
void resetWtd(void)
{
  unsigned char retval = 0x0;

  retval = readLogDevByte(WTD_DEV, SIO_WTD_STATUS);
  retval = retval & SIO_WTD_RESTO;
  writeLogDevByte(WTD_DEV, SIO_WTD_STATUS, retval);

  return;
}

/* -------------------------- API -------------------------- */

/*
 * Watchdog API
 */

ARB6002_API u8 getWtdTimer(void)
{
  static int wtdSetup = 0;
  unsigned char retval = 0x0;
  BOOL winIoInitOk = FALSE;

  /* WinIO Initialization */
  if(!winIoReady ) {
	  winIoInitOk = InitializeWinIo();

	  /* If WinIo32.sys initialization fails! */
	  if( !winIoInitOk ) {
		printf("InitializeWinIo() Fail!\n");
		return(ERR_INIT_WINIO);
	  }

	  winIoReady = TRUE;
  }


  if( !wtdSetup ) {
    setupWtd();
    wtdSetup = 1;
  }
    
  retval = readLogDevByte(WTD_DEV, SIO_WTD_COUNT);

  return(retval);
}


ARB6002_API void setWtdTimer(u8 value)
{
  static int wtdSetup = 0;
  BOOL winIoInitOk = FALSE;

  /* WinIO Initialization */
  if(!winIoReady ) {
	  winIoInitOk = InitializeWinIo();

	  /* If WinIo32.sys initialization fails! */
	  if( !winIoInitOk ) {
		printf("InitializeWinIo() Fail!\n");
		return;
	  }

	  winIoReady = TRUE;
  }

  if( !wtdSetup ) {
    setupWtd();
    wtdSetup = 1;
  }

  resetWtd();
  writeLogDevByte(WTD_DEV, SIO_WTD_COUNT, value);

  return;
}

/*
 * GPIO API
 */
ARB6002_API i32 getInChLevel(i32 channel, u8 *val )
{

  static int gpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask   = 0;
  BOOL winIoInitOk = FALSE;

  /* WinIO Initialization */
  if(!winIoReady ) {
	  winIoInitOk = InitializeWinIo();

	  /* If WinIo32.sys initialization fails! */
	  if( !winIoInitOk ) {
		printf("InitializeWinIo() Fail!\n");
		return(ERR_INIT_WINIO);
	  }

	  winIoReady = TRUE;
  }


  if( (channel < 0x0) || (channel > 0xf) ) {
    return(-1);
  }

  /* Check if GPIO has been configured properly. */
  if(!gpioSetup) {
    setupGpio();
    gpioSetup = 1;
  }


  /* The Input bits are bit 4 ~ bit 7 */
  mask = (unsigned char) channel << 4;

  retval = readLogDevByte(GPIO_DEV, SIO_GPIO_DATA);

  retval = retval & mask;
  retval = retval >> 4;

  *val = retval;

  return(0);
};

ARB6002_API i32 setOutChLevel(i32 channel, u8 val)
{
  static int gpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask = 0;
  BOOL winIoInitOk = FALSE;

  /* WinIO Initialization */
  if(!winIoReady ) {
	  winIoInitOk = InitializeWinIo();

	  /* If WinIo32.sys initialization fails! */
	  if( !winIoInitOk ) {
		printf("InitializeWinIo() Fail!\n");
		return(ERR_INIT_WINIO);
	  }

	  winIoReady = TRUE;
  }

  /* The Output bits are bit 0 ~ bit 3 */
  if( channel < 0 || channel > 0xf ) {
    return(-1);
  }

  /* Bits can be set to 0 or 1 only.
     Other value is wrong. */
  if( (val != 0) && (val != 1) ) {
    return(-1);
  }
  
  /* Check if GPIO has been configured properly. */
  if(!gpioSetup) {
    setupGpio();
    gpioSetup = 1;
  }

  mask = (unsigned char) channel & 0xff;
  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_DATA );

  if( val== 0 ) 
    retval = retval & ~mask;  // Set indicated bits to 0
  else
    retval = retval | mask;   // Set indicated bits to 1

  writeLogDevByte(GPIO_DEV, SIO_GPIO_DATA, retval);

  return(0);

}

ARB6002_API i32 getOutChLevel(i32 channel, u8 *val)
{
  static int gpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask   = 0;
  BOOL winIoInitOk = FALSE;

  /* WinIO Initialization */
  if(!winIoReady ) {
	  winIoInitOk = InitializeWinIo();

	  /* If WinIo32.sys initialization fails! */
	  if( !winIoInitOk ) {
		printf("InitializeWinIo() Fail!\n");
		return(ERR_INIT_WINIO);
	  }

	  winIoReady = TRUE;
  }

  if( channel < 0x0 || channel > 0xf ) {
    return(-1);
  }

  /* Check if GPIO has been configured properly. */
  if(!gpioSetup) {
    setupGpio();
    gpioSetup = 1;
  }

  /* The Output bits are bit 0 ~ bit 3 */
  mask = (unsigned char) channel;

  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_DATA );

  retval = retval & mask;

  *val = retval;

  return(0);
}

BOOL APIENTRY DllMain( HMODULE hModule,
                       DWORD  ul_reason_for_call,
                       LPVOID lpReserved
					 )
{
	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
		break;
	case DLL_THREAD_ATTACH:
	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		releaseCard();
		break;
	}
    return TRUE;
}
