#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>
#include <unistd.h> // For usleep()
#include <errno.h>
#include "pwr_acce.h"

  
static struct PwrDev pwrDev = {
        .port = "/dev/ttyS5", // The power subsystem connects to COM6.
        .fd   = 0,
        .init = 0,
};


//#define ERR_MSG

#ifdef ERR_MSG
# define EMSG(msg) fprintf( stderr, msg)
#endif


#define MUTEX

#ifdef MUTEX
static pthread_mutex_t accessing = PTHREAD_MUTEX_INITIALIZER;
#endif

static int initPort(void);
static int doHandShake(void);
static int readPort(unsigned char *buf, size_t size);
static int writePort(unsigned char *buf, size_t size);

int setPicDefault(void)
{
  int result = -1;
  unsigned char cmd;

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_PIC_DEFAULT;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

//  result = -1;
//  result = writePort( setTime, 3);
//  if( result == -1 ) {
//#ifdef MUTEX
//    pthread_mutex_unlock( &accessing );
//#endif
//    return(-1);
//  }
//
//  /*
//   * The system waits for 1 mini-second after writing data,
//   * before reading the returned value.
//   */
//  usleep(1000);
//  
//  /* Note:
//   *  The Power subsystem will return
//   *  a code after writing the 'setTime' data to the Power subsystem.
//   *  If the data is written to the Power subsystem successfully, the
//   *  returned code is 0x0. We MUST read the code out of the serial
//   *  port buffer, or the code will become the returned code in the
//   *  next hand-shake stage. This will cause an error.
//   */  
  result = -1;
  cmd = 0xff;

  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif
       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


/* Get the Ignition status */
int getIgnStatus( unsigned char *ignStatus)
{
  int result = -1;
  unsigned char cmd;
  unsigned char temp[2] = {0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_IGN_STS;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  /* Note:
   * The Power subsystem returns an extra byte besides the 1 byte
   * 'powerOnMode' data to indicate if the data has been successfully
   * returned.
   */
  result = -1;
  result = readPort( temp, 2);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  if( temp[1] == 0x0 ) {
      *ignStatus = temp[0];
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


int setSoftOffDelayS( unsigned long setTime )
{
  int result = -1;
  unsigned char cmd;
	unsigned char temp[3]={0};

  if( setTime>255 )
    return(-1);		

  setTime=setTime*10;

	temp[0]=(unsigned char) (setTime & 0xff);
  temp[1]=(unsigned char) ( (setTime>>8) & 0xff);
	temp[2]=(unsigned char) ( (setTime>>16) & 0xff);

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_SOFTOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  result = -1;
  result = writePort( temp, 3);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /*
   * The system waits for 1 mini-second after writing data,
   * before reading the returned value.
   */
  usleep(1000);
  
  /* Note:
   *  The Power subsystem will return
   *  a code after writing the 'setTime' data to the Power subsystem.
   *  If the data is written to the Power subsystem successfully, the
   *  returned code is 0x0. We MUST read the code out of the serial
   *  port buffer, or the code will become the returned code in the
   *  next hand-shake stage. This will cause an error.
   */  
  result = -1;
  cmd = 0xff;

  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif
       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int setSoftOffDelayM( unsigned long setTime )
{
  int result = -1;
  unsigned char cmd;
	unsigned char temp[3]={0};

  if( setTime>255 )
    return(-1);		

  setTime=setTime*10*60;

	temp[0]=(unsigned char) (setTime & 0xff);
  temp[1]=(unsigned char) ( (setTime>>8) & 0xff);
	temp[2]=(unsigned char) ( (setTime>>16) & 0xff);

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_SOFTOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  result = -1;
  result = writePort( temp, 3);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /*
   * The system waits for 1 mini-second after writing data,
   * before reading the returned value.
   */
  usleep(1000);
  
  /* Note:
   *  The Power subsystem will return
   *  a code after writing the 'setTime' data to the Power subsystem.
   *  If the data is written to the Power subsystem successfully, the
   *  returned code is 0x0. We MUST read the code out of the serial
   *  port buffer, or the code will become the returned code in the
   *  next hand-shake stage. This will cause an error.
   */  
  result = -1;
  cmd = 0xff;

  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif
       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


int setHardOffDelayS( unsigned long setTime )
{
  int result = -1;
  unsigned char cmd;
  unsigned char temp[3]={0};

  if( setTime>255 )
		return(-1);

  setTime=setTime*10;

	temp[0]=(unsigned char) (setTime & 0xff);
  temp[1]=(unsigned char) ( (setTime>>8) & 0xff);
	temp[2]=(unsigned char) ( (setTime>>16) & 0xff);


  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_HARDOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  result = -1;
  result = writePort( temp, 3);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /*
   * The system waits for 1 mini-second after writing data,
   * before reading the returned value.
   */
  usleep(1000);

  /* Note:
   *  The Power subsystem will return
   *  a code after writing the 'setTime' data to the Power subsystem.
   *  If the data is written to the Power subsystem successfully, the
   *  returned code is 0x0. We MUST read the code out of the serial
   *  port buffer, or the code will become the returned code in the
   *  next hand shake stage. This will cause an error.
   */  
  result = -1;
  cmd = 0xff;
  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif
       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int setHardOffDelayM( unsigned long setTime )
{
  int result = -1;
  unsigned char cmd;
  unsigned char temp[3]={0};

  if( setTime>255 )
		return(-1);

  setTime=setTime*10*60;

	temp[0]=(unsigned char) (setTime & 0xff);
  temp[1]=(unsigned char) ( (setTime>>8) & 0xff);
	temp[2]=(unsigned char) ( (setTime>>16) & 0xff);


  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_HARDOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  result = -1;
  result = writePort( temp, 3);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /*
   * The system waits for 1 mini-second after writing data,
   * before reading the returned value.
   */
  usleep(1000);

  /* Note:
   *  The Power subsystem will return
   *  a code after writing the 'setTime' data to the Power subsystem.
   *  If the data is written to the Power subsystem successfully, the
   *  returned code is 0x0. We MUST read the code out of the serial
   *  port buffer, or the code will become the returned code in the
   *  next hand shake stage. This will cause an error.
   */  
  result = -1;
  cmd = 0xff;
  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif
       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


int setPowerOnMode( unsigned char powerOnMode )
{
  int result = -1;
  unsigned char cmd;

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_SET_PWR_MOD;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep(USLEEP);

  result = -1;
  result = writePort( &powerOnMode, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif

    return(-1);
  }

  /*
   * The system waits for 1 mini-second after writing data,
   * before reading the returned value.
   */
  usleep(1000);

  /* Note:
   *  The Power subsystem will return
   *  a code after writing the 'setTime' data to the Power subsystem.
   *  If the data is written to the Power subsystem successfully, the
   *  returned code is 0x0. We MUST read the code out of the serial
   *  port buffer, or the code will become the returned code in the
   *  next hand shake stage. This will cause an error.
   */  
  result = -1;
  cmd = 0xff;
  result = readPort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif

    return(-1);
  } else {
    if( cmd != 0x0 ) {
#ifdef MUTEX
      pthread_mutex_unlock( &accessing );
#endif

       return(-1);
    }
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


int getSoftOffDelay( unsigned long *timeDelay )
{
  int i;
  int result = -1;
  unsigned char cmd;
  unsigned char temp[4] = {0xff, 0xff, 0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_SOFTOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 3 bytes 'setTime' data to indicate if the data has been
   * correctly returned.
   */
  result = -1;
  result = readPort( temp, 4);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 4th byte equals 0x0, the data at temp[0] ~ temp[2] are correct. */
  if( temp[3] == 0x0 ) {
//    for(i=0; i<3; i++)
//      setTime[i] = temp[i];
			*timeDelay = 0;
			*timeDelay = (temp[2]<<16)|(temp[1]<<8)|(temp[0]);
      *timeDelay = *timeDelay/10;
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int getHardOffDelay( unsigned long *timeDelay )
{
  int i;
  int result = -1;
  unsigned char cmd;
  unsigned char temp[4] = {0xff, 0xff, 0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_HARDOFF;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 3 bytes 'setTime' data to indicate if the data has been
   * correctly returned.
   */
  result = -1;
  result = readPort( temp, 4);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 4th byte equals 0x0, the data at temp[0] ~ temp[2] are correct. */
  if( temp[3] == 0x0 ) {
//    for(i=0; i<3; i++)
//      setTime[i] = temp[i];
			*timeDelay = 0;
			*timeDelay = (temp[2]<<16)|(temp[1]<<8)|(temp[0]);
      *timeDelay = *timeDelay/10;
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int getPicMode( unsigned char *picMode )
{
  int result = -1;
  unsigned char cmd;
  unsigned char temp[2] = {0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_PIC_MOD;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 1 bytes 'powerOnMode' data to indicate if the data has been
   * correctly returned.
   */

  result = -1;
  result = readPort( temp, 2);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 2nd byte equals 0x0, the data at temp[0] is correct. */
  if( temp[1] == 0x0 ) {
      *picMode = temp[0];
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }


#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}


int getPowerOnMode( unsigned char *powerOnMode )
{
  int result = -1;
  unsigned char cmd;
  unsigned char temp[2] = {0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_PWR_MOD;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 1 bytes 'powerOnMode' data to indicate if the data has been
   * correctly returned.
   */

  result = -1;
  result = readPort( temp, 2);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 2nd byte equals 0x0, the data at temp[0] is correct. */
  if( temp[1] == 0x0 ) {
      *powerOnMode = temp[0];
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }


#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int getBattVolt( unsigned char *volt )
{
  int i;
  int result = -1;
  unsigned char cmd;
  unsigned char temp[3] = {0xff, 0xff, 0xff};

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_BAT_VOL;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 2 bytes 'Battery Voltage' data to indicate if the data has been
   * correctly returned.
   */
  result = -1;
  result = readPort( temp, 3);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 3th byte equals 0x0, the data at temp[0] ~ temp[1] are correct. */
  if( temp[2] == 0x0 ) {
    for(i=0; i<2; i++)
      volt[i] = temp[i];
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int getPicFwVer( PicInfo *ver )
{
  int i;
  int result = -1;
  unsigned char cmd;
  unsigned char *pVer=(unsigned char *)ver;
  unsigned char temp[10] = {0xff, 0xff,
                            0xff, 0xff,
                            0xff, 0xff,
                            0xff, 0xff,
                            0xff, 0xff,
                           };

  if( pwrDev.init == 0 ) {
    result = initPort();
    if( result == -1 )
      return(-1);
  }
    
#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  cmd = CMD_GET_PIC_VER;

  result = -1;
  result = writePort( &cmd, 1);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  result = doHandShake();
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  usleep( USLEEP );

  /* Note:
   * The Power subsystem returns an extra
   * byte besides the 9 bytes 'PIC Firmware Version' information to indicate if the data has been
   * correctly returned.
   */
  result = -1;
  result = readPort( temp, 10);
  if( result == -1 ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* If the 3th byte equals 0x0, the data at temp[0] ~ temp[1] are correct. */
  if( temp[9] == 0x0 ) {
    for(i=0; i<9; i++)
      pVer[i] = temp[i];
  } else {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

/* --- API Code End --- */



/*
 * Configure the COM port to:
 *  1. Character size = 8 bits
 *  2. Baud rate = 19200
 *  3. Stop bit = 1
 *  4. No parity check
 *  5. Other items.
 */
int initPort(void)
{
  int result = -1;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  pwrDev.fd = open( pwrDev.port, O_RDWR|O_NONBLOCK|O_NOCTTY);
  if( pwrDev.fd == -1 ) {
#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* Get the termios of the port */
  result = tcgetattr( pwrDev.fd, &pwrDev.portios );
  if( result == -1 ) {

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* Set Baud rate to 19200 */
  result = -1;
  // input baud rate
  result = cfsetispeed( &pwrDev.portios, B19200 );
  if( result == -1 ) {

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  result = -1;
  // output baud rate
  result = cfsetospeed( &pwrDev.portios, B19200 );
  if( result == -1 ) {

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }


  /* Set control mode to CS8 */
  pwrDev.portios.c_cflag |= (CS8|CREAD|CLOCAL|HUPCL);

  /* Set control mod to 1 stop bit */
  pwrDev.portios.c_cflag &= ~CSTOPB;

  /* Disable parity check */
  pwrDev.portios.c_cflag &= ~PARENB;

  /* Extra Settings */
  pwrDev.portios.c_iflag = IGNPAR;
  pwrDev.portios.c_oflag = 0;
  pwrDev.portios.c_lflag = 0;        //ICANON;

  pwrDev.portios.c_cc[VMIN] = 1;
  pwrDev.portios.c_cc[VTIME] = 0;

  /* Implement all the configurations */
  result = -1;
  result = tcsetattr( pwrDev.fd, TCSANOW, &pwrDev.portios );
  if( result == -1 ) {

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  pwrDev.init = 1;

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
}

int doHandShake(void)
{
  unsigned char handShake = 0x0;
  int           result    = -1;

  result = readPort( &handShake, 1 );
  if( result == -1 ) {
    return(-1);
  }

  /* If hand-shake responds, it will return the code '0xaa' */  
  if( handShake != 0xaa ) {
    return(-1);
  } else {

    result = -1;
    handShake = 0x55;

    /* Return '0x55' to finish the hand-shake. */
    result = writePort( &handShake, 1 );
    if( result == -1 ) {
      return(-1);
    }
  }

  return(0);
}

int readPort( unsigned char *buf, size_t size)
{
  int times          = TIMES;
  int totalFinished  = 0;
  int numFinished    = 0;

  while( totalFinished < size ) {

    numFinished = read( pwrDev.fd, buf+totalFinished, size-totalFinished );
    if( numFinished == -1 ) {
      if( errno == EAGAIN ) {
        usleep(USLEEP);
        times--;
        numFinished = 0;
      } else {
        return(-1);
      }
    }

    if( times  == 0 ) {
      return(-1);
    }

    totalFinished += numFinished;
  }

  return(0);
}

int writePort( unsigned char *buf, size_t size)
{
  int times          = TIMES;
  int totalFinished = 0;
  int numFinished    = 0;

  while( totalFinished < size ) {

    numFinished = write( pwrDev.fd, buf+totalFinished, size-totalFinished );

    if( numFinished == -1 ) {
      if( errno == EAGAIN ) {
        usleep(USLEEP);
        times--;
        numFinished = 0;
      } else {
        return(-1);
      }
    }

    if( times  == 0 )
      return(-1);

    totalFinished += numFinished;
  }

  return(0);
}
